function f2_extract_bodies_from_IJ(all_paths)
 
% Description.
% - Shows 3D segmentation done elsewhere (e.g. ImageJ) as 2D blob = a slice through a body 
%   at a given z
% - allow the user to connect these 2D blobs (body slices) in 3D. This was
%   done as an alternative for "bwconncomp"; because manual segmentatoin is
%   noisy and layers can overlap and thus connect two bodies that are
%   supposed to be separate. In manual conneciton of these layers this will
%   not happen.
% - Saves binary Matrices as images 
%
% Dmitry Ershov 
% 2017 October


%% Parameters 
% hard coded.
params.image_digit_format                  = 4;                        % number of digits in the image file name; padded with 0; i.e.: 'shape0015.TIF'
params.image_format                        = 'TIF';                    % image format
params.image_size                          = [];                       % calculated below. Same as the experimental stack.
    
params.connectivity_2D                     = 4;                        % pixel connectivity; 4 is side-to-side pixel connectivity.
params.connectivity_3D                     = 6;                        % voxel connectivity; 6 is face-to-face voxel connectivity.
params.area_thresh_2D                      = 3;                        % are threshold: remove small lost (disconnected) 2D-blobs (junk)
params.polar_domain_cortex_thickness       = 2;                        % 2-3 is enough. should be more than 1!



%% Check if the labelling has already been done.
file_name           = 'labeled_bodies.mat';
file_path           = [all_paths.shapes_source, file_name];

if exist(file_path, 'file') 
    disp(' ')
    disp(['Found existing labeled bodies data; loading files: ']);
    disp(['    ', file_path]);
    disp(['    ', 'To extract anew, delete manually the file: ', file_name]);

    load(file_path, 'all_bodies_labeled_uint8')
    source_body_stack   = all_bodies_labeled_uint8;
    clear all_bodies_labeled_uint8;
else
    % if does not exist: make an empty holder.
    source_body_stack = [];
end



% if labeling does not exist (= empty matrix), load raw segmentation:
if isempty(source_body_stack)

    % Open the experimental segmentaiton
    [filename, path_name] = uigetfile([all_paths.shapes_source, '*.*'], 'Select a source shape image stack');
    file_abs_path = [path_name, filename];

    % check format
    try
        inf = imfinfo(file_abs_path);
    catch
        disp(' ')
        disp('Could not determine file format')
        disp('Make sure you are selecting an image stack.')
        disp('Quitting.')
        return;
    end

    % read in the images
    for i = 1 : length(inf)    
        cur_img             =  imread(file_abs_path, i);
        source_body_stack   = cat(3, source_body_stack, cur_img); 
    end

    % invert so that the cell is 255, background is 0.
    source_body_stack = uint8(~source_body_stack);

    % ADDITIONAL CLEANING
    % you may introduce your own specific cleaning
    for i = 1:size(source_body_stack,3)

        % clean spurious pixels:
       source_body_stack(:,:,i) =  bwmorph(source_body_stack(:,:,i), 'spur');

       % clean lone pixels:
       source_body_stack(:,:,i) =  bwmorph(source_body_stack(:,:,i), 'clean');

       % clean H-connections:
       source_body_stack(:,:,i) =  bwmorph(source_body_stack(:,:,i), 'hbreak');

       % diagonal fill to eliminate 8-connectivity of the background
       % source_body_stack(:,:,i) =  bwmorph(source_body_stack(:,:,i), 'diag');
    end
end


fig = makeFig;

params.image_size   = size(source_body_stack);

% user data:
uD.all_paths        = all_paths;
uD.A                = source_body_stack;    % here we store original img
uD.B                = source_body_stack;    % here we will change pixel values
uD.current_label    = 2;
uD.cur_fr           = round(size(source_body_stack, 3)/2);
uD.max_fr           = size(source_body_stack, 3);
uD.show_labels      = 1;
uD.connectivity_2D  = params.connectivity_2D;
uD.connectivity_3D  = params.connectivity_3D;
uD.area_thresh_2D = params.area_thresh_2D; % area in pixels


set(fig, 'UserData', uD);
update_axes_1(fig)
update_axes_2(fig)


end





function callback_button(fig, evt)
% label the clicked region

uD = get(fig, 'UserData');

cur_ax = get(fig, 'CurrentAxes');
cur_point = get(cur_ax, 'CurrentPoint');
x = round(cur_point(1,1));
y = round(cur_point(1,2));

% disp([num2str(x), ',' num2str(y)])

b = bwlabel(uD.A(:,:, uD.cur_fr), uD.connectivity_2D);     
r = regionprops(b, 'PixelList', 'PixelIdxList');

for k = 1 : length(r)
    % plot(r(k).PixelList(:,1), r(k).PixelList(:,2),'w.')
    if ismember([x, y], r(k).PixelList, 'rows')
        c = uD.B(:,:, uD.cur_fr);
        c(r(k).PixelIdxList) = uD.current_label;
        uD.B(:,:, uD.cur_fr) = c;
        disp(['setting region values to ' num2str(uD.current_label)])
    end
end

uD.cur_fr = uD.cur_fr + 1;
uD = check_fr_limit(uD);

set(fig, 'UserData', uD);
update_axes_1(fig)

end
 



function callback_key(fig, evt)
uD = get(fig, 'UserData');

switch evt.Key
    
    case 'l'
        % show\hide ROI labels
        uD.show_labels = ~uD.show_labels;
        
        
    case 'c'
        % clean bodies from small pixels
        uD.B = clean_labeled_bodies(uD.B, uD.area_thresh_2D, uD.connectivity_3D);        
        
    case 'v'
        % recalculate the 3D view
        update_axes_2(fig)
      
      
    case 'a'
        % go to first frame
        uD.cur_fr = 1;
    
    case 's'
        % uD.B = edited (connected) 2D slices
        save_labeled_bodies(uD.B, uD.all_paths);       
        
        
    otherwise
        % check if number
        [num, status] = str2num(evt.Key);
        if status
            
            disp(['Current label ' num2str(num)]);

            uD = get(fig, 'UserData');
            uD.current_label = num;

        end
end

set(fig, 'UserData', uD);
update_axes_1(fig)

end


function save_labeled_bodies(labeled_bodies, all_paths)

all_bodies_labeled_uint8 = labeled_bodies;

% saving
file_name           = 'labeled_bodies.mat';
file_path           = [all_paths.shapes_source, file_name];
disp(' ')
disp(['Saving the matrix with labeled bodies: ']);
disp(['    ', file_path]);
save(file_path, 'all_bodies_labeled_uint8');
disp(['    Done']);
        
end








function labeled_bodies_cleaned = clean_labeled_bodies(labeled_bodies, area_thresh_2D, connectivity_3D)

disp(' ')
disp('    Cleaning disconnected 2D areas...')

% go over each slice and remove 1or 2pixel regions.
for i = 1 : size(labeled_bodies, 3)
    
    b_inds = labeled_bodies(:,:,i);
    r = regionprops(b_inds, 'Area', 'PixelList', 'Centroid', 'PixelIdxList');
    
    for k = 1 : length(r)
       if r(k).Area < area_thresh_2D && r(k).Area > 0      % if a region is too small, remove it

           % can visualize:
           if 0
               figure;
               imagesc(b)
               hold on
               plot(r(k).PixelList(:,1), r(k).PixelList(:,2), 'w.')
               title('white = removed region')
           end
           
           % remove the small regions:
           b_inds(r(k).PixelIdxList) = 0;
           labeled_bodies(:,:,i) = b_inds;
       end
    end
    
end


disp('    Cleaning disconnected 3D volumes...')

% there are might still be bodies that are disconnected.
% go over each 3D body and check if it is one piece.
% if not: clean it from small junk
labeled_bodies_cleaned = uint8(zeros(size(labeled_bodies)));

b_inds              = unique(labeled_bodies);
b_inds(b_inds == 0) = [];

for b_i = b_inds'
    
    B_i = labeled_bodies;
    B_i(labeled_bodies ~= b_i) = 0;     

    cc              = bwconncomp(B_i, connectivity_3D);
    
    while cc.NumObjects > 1 
        % more than one body: there might be some junk largest body.
        % find the largest body, remove the rest.
       
        max_volume_pix = 0;
        
        for ll = 1 : cc.NumObjects
            max_volume_pix = max([max_volume_pix, length(cc.PixelIdxList{ll})]);
        end
        
        % now set the smallest pieces to 0:
        for ll = 1 : cc.NumObjects
            if ~(length(cc.PixelIdxList{ll}) == max_volume_pix)
                B_i(cc.PixelIdxList{ll}) = 0;
            end
        end
        
        cc              = bwconncomp(B_i, 6);
    end
    
    % now B_i is clean, collect it:
    labeled_bodies_cleaned(logical(B_i)) = b_i;
    
end
  
% disp('Done.')
end







function update_axes_1(fig)
figure(fig);
uD = get(fig, 'UserData');


body_count = unique(uD.B);
body_count(body_count == 0) = [];
col_set = jet(length(body_count) + 1);
% I shift the color here to match the colormap from the axes 1;
% because they have +1 ROI, which background.


subplot(1, 2, 1)
cla;

col_set(1,:) = 0.8*[1 , 1, 1]; % set BG to grey
colormap(col_set)

title({['Frame: ' num2str(uD.cur_fr)],...
        ['Current Label: ' num2str(uD.current_label)]})
     

imagesc(uD.B(:,:, uD.cur_fr))

r = regionprops(uD.B(:,:, uD.cur_fr), 'Centroid', 'PixelIdxList');% 'MaxIntensity');

for i = 1 : length(r)
   x = r(i).Centroid(1);
   y = r(i).Centroid(2);
   
   b = uD.B(:, :, uD.cur_fr);
   v = unique(b(r(i).PixelIdxList));
   
   if uD.show_labels
       text(x, y, num2str(v), 'color', 'w', 'background', 'k')
   end
   
end

     
end




function update_axes_2(fig)

uD = get(fig, 'UserData');

subplot(1, 2, 2)
cla;


body_count = unique(uD.B);
body_count(body_count == 0) = [];
col_set = jet(length(body_count) + 1);
% I shift the color here to match the colormap from the axes 1;
% because they have +1 ROI, which background.

for i = 1 : length(body_count)
    
    body_ind = body_count(i);

    C = uD.B;
    C(C ~= body_ind) = 0; % remove the values that are not equal to the current body index
    C(C == body_ind) = 1;

    patch(isosurface(C, 0.5),...
        'FaceColor', col_set(i+1,:), 'FaceAlpha', 0.35, 'EdgeColor', 'none'); 
    indx = find(C == 1);
    [r,c,z] = ind2sub(size(C), indx);
    x = mean(c);
    y = mean(r);
    z = mean(z);
    
    if uD.show_labels
        text(x,y,z, num2str(body_ind), 'color', 'w', 'background', 'k')
    end
end
       

end





 



function fig = makeFig
fig = figure('position', [  150         150        1600         500],...
    'WindowScrollWheelFcn', @callback_wheel,...
    'WindowButtonDownFcn', @callback_button,...
     'WindowKeyPressFcn', @callback_key);

 colormap jet;
 
subplot(1, 2, 1)
hold on;
axis equal
axis tight
xlabel('X'); ylabel('Y');
set(gca, 'YDir', 'reverse') % reversed Y -> typical for displaying matrices/images


subplot(1, 2, 2)
hold on;
axis equal;
box on;
xlabel('X'); ylabel('Y'); zlabel('Z')
set(gca, 'YDir', 'reverse')
view([30 30])
end

 




function callback_wheel(fig, evt)
uD = get(fig, 'UserData');

uD.cur_fr = uD.cur_fr + evt.VerticalScrollCount;
uD = check_fr_limit(uD);

set(fig, 'UserData', uD);
update_axes_1(fig)
end





function uD = check_fr_limit(uD)

if uD.cur_fr < 1
    uD.cur_fr = 1;
end
if uD.cur_fr > uD.max_fr
    uD.cur_fr = uD.max_fr;
end

end





